//*************************************************************************************************
//
//	Description:
//		PostProcessing.fx - Various techniques and shaders which are used to perform Post Processing
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Barry Irvine
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		BIrvine         26/03/2007  0.1           Created
//		TMann						22/06/2007	0.2					  PS3 support
//		TNettleship     07/24/2007  0.3	          Made sure samplers aren't using anisotropic filtering.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"

texture tApplyTexture : TEXTURE;

sampler sApplySampler : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < tApplyTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

//

texture tCopyTexture : TEXTURE;

sampler sCopySampler : SAMPLER = sampler_state
{
#ifdef _PS3_
	FX_SAMPLERSTATE_SRGB_TEXTURE
#else
	FX_SAMPLERSTATE_LINEAR_TEXTURE
#endif
	Texture = < tCopyTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};


texture tSceneTexture : TEXTURE;
sampler sSceneSampler : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < tSceneTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
};

texture tAccumulatedDistortionTexture : TEXTURE;
sampler sAccumulatedDistortionSampler : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < tAccumulatedDistortionTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
};

texture tAccumulatedDistortionNormalTexture : TEXTURE;
sampler sAccumulatedDistortionNormalSampler : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < tAccumulatedDistortionNormalTexture >;
	AddressU  = Wrap;
	AddressV  = Wrap;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
};

texture tBlurTexture : TEXTURE;
sampler sBlurSampler : SAMPLER = sampler_state
{
#ifdef _PS3_
	FX_SAMPLERSTATE_SRGB_TEXTURE
#else
	FX_SAMPLERSTATE_LINEAR_TEXTURE
#endif
	Texture = < tSceneTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
};


/*************************************************************************************************
* SHADER INPUT/OUPUT STRUCTURES
*************************************************************************************************/

struct A2V_PREOVERLAY_APPLY
{
#ifdef _PS3_
	float2 Position : POSITION;
#else
	float3 Position : POSITION;
	float2 TexCoord : TEXCOORD0;
#endif	
};

struct V2P_PREOVERLAY_APPLY
{
	float4 Position : POSITION;
	float2 TexCoord : TEXCOORD0;
	float2 BlendPos : TEXCOORD1;
};

//

struct A2V_PREOVERLAY_COPY
{
#ifdef _PS3_
	float2 Position : POSITION;
#else
	float3 Position : POSITION;
	float2 TexCoord : TEXCOORD0;
#endif
};

struct V2P_PREOVERLAY_COPY
{
	float4 Position : POSITION;
	float2 TexCoord : TEXCOORD0;
};

struct VSOUTPUT_HEATHAZE
{
	float4 Position : POSITION;
	float2 TexCoord : TEXCOORD0;
	float2 TexCoordScroll : TEXCOORD1;
};

//

struct P2T_PREOVERLAY
{
	COLOUR_OUTPUT_TYPE Colour0 : COLOR0;
};

/*************************************************************************************************
* SETTINGS
*************************************************************************************************/

	float gOffsetScale;
	float gBlendLimit;
	float gBlendSize;
	float gBlendPosition;

	float currentTime;
	float2 sceneTextureWidthHeight;
	float distortionIntensity = 0.015f;

/*************************************************************************************************
*************************************************************************************************/
V2P_PREOVERLAY_APPLY PreOverlayApply_VS( A2V_PREOVERLAY_APPLY lInput )
{
	V2P_PREOVERLAY_APPLY lOutput;
	float2 TexCoord;


	//
	
#ifdef _PS3_
	lOutput.Position	=	float4( lInput.Position.xy, 0.0f, 1.0 );
	// Generates its own coords
	TexCoord.x=lInput.Position.x*0.5f+0.5f;
	TexCoord.y=lInput.Position.y*-0.5f+0.5f;
#else
	lOutput.Position	=	float4( lInput.Position.xyz, 1.0 );
	TexCoord=lInput.TexCoord;
#endif		

	lOutput.TexCoord =  normalize(TexCoord - 0.5) * gOffsetScale;
	lOutput.TexCoord += TexCoord;

	//

  lOutput.BlendPos = 2.0f * (TexCoord - 0.5);	// transform into -1 to +1 range

#ifndef _PS3_  
  // Why is this upside down?
  lOutput.TexCoord.y = 1.0f - lOutput.TexCoord.y;
#endif

	lOutput.BlendPos *= rsqrt( 2.0f );									// scale so that the length of the vector from the centre to the corner of the screen is 1

	return lOutput;
}

/*************************************************************************************************
*************************************************************************************************/
P2T_PREOVERLAY PreOverlayApply_PS( V2P_PREOVERLAY_APPLY lInput )
{
	P2T_PREOVERLAY lOutput;

	lOutput.Colour0.xyz = tex2D(sApplySampler, lInput.TexCoord).xyz;
//lOutput.Colour0.xyz *= float3(0.5,0.5,0.5);			// vignette test

	float	lBlend = length( lInput.BlendPos );

	lBlend	*=	gBlendSize;												// scale the size of the fade area
	lBlend	+=	gBlendPosition;												// move the fade position
	lBlend	=	saturate( lBlend );						// clamp to 0~1 range

	lOutput.Colour0.w = lBlend * gBlendLimit;	// Apply max blend scaler

/*
	if (lBlend == 0)
		lOutput.Colour0.xyz = float3(1,0,0);
	else if (lBlend == 1)
		lOutput.Colour0.xyz = float3(0,1,0);
	else
		lOutput.Colour0.xyz = lBlend;

	lOutput.Colour0.w = 1;
*/

	return lOutput;
}

/*************************************************************************************************
*************************************************************************************************/
V2P_PREOVERLAY_COPY PreOverlayCopy_VS( A2V_PREOVERLAY_COPY lInput )
{
	V2P_PREOVERLAY_COPY lOutput;

	lOutput.Position	=	float4( lInput.Position.xy, 0.0f, 1.0f );
	lOutput.TexCoord.x=lInput.Position.x*0.5f+0.5f;
	lOutput.TexCoord.y=lInput.Position.y*-0.5f+0.5f;

	return lOutput;
}

/*************************************************************************************************
*************************************************************************************************/
P2T_PREOVERLAY PreOverlayCopy_PS( V2P_PREOVERLAY_COPY lInput )
{
	P2T_PREOVERLAY lOutput;

	lOutput.Colour0 = tex2D(sCopySampler, lInput.TexCoord);

	return lOutput;
}

/*************************************************************************************************
*************************************************************************************************/
P2T_PREOVERLAY DownsampleWithBlur_PS( V2P_PREOVERLAY_COPY lInput )
{
	P2T_PREOVERLAY lOutput;

	lOutput.Colour0 = tex2D(sBlurSampler, lInput.TexCoord);

	return lOutput;
}


/*************************************************************************************************
*************************************************************************************************/
VSOUTPUT_HEATHAZE AccumulatedDistortion_VS( A2V_PREOVERLAY_COPY lInput )
{
	VSOUTPUT_HEATHAZE lOutput;

	lOutput.Position	=	float4( lInput.Position.xy, 0.0f, 1.0f );
	lOutput.TexCoord.x = lInput.Position.x*0.5f+0.5f;
	lOutput.TexCoord.y = lInput.Position.y*-0.5f+0.5f;

	lOutput.TexCoordScroll = lOutput.TexCoord*4.0f;	// tile it
	lOutput.TexCoordScroll.y += currentTime * 0.7f;	// scroll it

	lOutput.TexCoord += 0.5f/sceneTextureWidthHeight.xy;	// adjust tex coords to align to screen

	return lOutput;
}

/*************************************************************************************************
*************************************************************************************************/
P2T_PREOVERLAY AccumulatedDistortion_PS( VSOUTPUT_HEATHAZE lInput )
{
	P2T_PREOVERLAY lOutput;

	float	accumulated_amount = tex2D( sAccumulatedDistortionSampler, lInput.TexCoord).r;
	float3	normal = tex2D( sAccumulatedDistortionNormalSampler, lInput.TexCoordScroll);
	normal = (normal*2.0f)-1.0f;

	float2 offset = normal.xy;
	
	// calc projected screen space coords
	float2 distortTexCoords = lInput.TexCoord;
	float	distortAmount = 0.011f * accumulated_amount;	// 0.015  TODO: scale by screen resolution?
	distortTexCoords += offset*distortAmount;

	lOutput.Colour0 = tex2D( sSceneSampler, distortTexCoords);
	lOutput.Colour0.a = 1.0f;

//lOutput.Colour0.rgb += heathaze;
//lOutput.Colour0.rgb += ((normal+1.0f)*0.5f)*accumulated_amount;

	return lOutput;
}



/*************************************************************************************************
* Techniques
*************************************************************************************************/
technique PreOverlayApply
{
	pass Pass0
	{
#ifdef _PS3_
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaTestEnable = false;
		AlphaBlendEnable = true;
		BlendFunc=int2(SrcAlpha, OneMinusSrcAlpha);
		BlendEquation=int(FuncAdd);
		CullFaceEnable = false;
		VertexShader = compile sce_vp_rsx PreOverlayApply_VS();
		PixelShader = compile sce_fp_rsx PreOverlayApply_PS();
#else
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaTestEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
		CullMode = None;
		VertexShader = compile vs_3_0 PreOverlayApply_VS();
		PixelShader = compile ps_3_0 PreOverlayApply_PS();
#endif		
	}
}

technique PreOverlayCopy
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaTestEnable = false;
		AlphaBlendEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx PreOverlayCopy_VS();
		PixelShader = compile sce_fp_rsx PreOverlayCopy_PS();
#else
		CullMode = None;
		VertexShader = compile vs_3_0 PreOverlayCopy_VS();
		PixelShader = compile ps_3_0 PreOverlayCopy_PS();
#endif		
	}
}

technique DownsampleWithBlur
<
	int regcount=43;
>
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaTestEnable = false;
		AlphaBlendEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx PreOverlayCopy_VS();
		PixelShader = compile sce_fp_rsx DownsampleWithBlur_PS();
#else
		CullMode = None;
		VertexShader = compile vs_3_0 PreOverlayCopy_VS();
		PixelShader = compile ps_3_0 DownsampleWithBlur_PS();
#endif		
	}
}

technique AccumulatedDistortion
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaTestEnable = false;
		AlphaBlendEnable = false;
#ifdef _PS3_
		CullFaceEnable=false;
		VertexShader = compile sce_vp_rsx AccumulatedDistortion_VS();
		PixelShader = compile sce_fp_rsx AccumulatedDistortion_PS();
#else
		CullMode = None;
		VertexShader = compile vs_3_0 AccumulatedDistortion_VS();
		PixelShader = compile ps_3_0 AccumulatedDistortion_PS();
#endif		
	}
}
